/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.*;

/**
 * This sample shows how to add a sub-report with RDC API to a report. The Data Source Configuration "Sample Database"
 * is necessary for this sample. You can create it using Data Source Manager in i-net Designer or the Configuration
 * Manager.<BR>
 * <BR>
 * <B>Note</B>: You may to change the path to the picture before running this sample.
 */
public class SubreportSample extends RDCSample {

    // Modify this Path if the location of the picture is another than this.
    static final String PICTUREPATH = "samples/rdc/cc_logo.png";

    /**
     * Adds a sub-report to a new report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create an Engine for a new report 
            Engine eng = RDC.createEmptyEngine( exportFmt );

            // Fill the engine
            // Get an area and a section to fill in some elements
            Area area = eng.getArea( 2 );
            Section section = area.getSection( 0 );

            // Add a text to the main report
            Text text = section.addText( 4000, 5100, 3000, 500 );
            Paragraph para = text.addParagraph();
            para.addTextPart( "This picture is in the main report." );

            // Add a picture to the main report
            URL pictureURL = getClass().getResource( '/' + PICTUREPATH );
            String path;
            if( pictureURL != null ) {
                path = pictureURL.getPath();
            } else {
                path = PICTUREPATH;
            }
            section.addPicture( 4500, 5500, 2835, 375, path );

            // This adds a new report as subreport
            Subreport subrep = section.addSubreport( 1000, 100, 6000, 700 );

            // Make subreport onDemand
            // subrep.setOnDemand(true);
            subrep.setOnDemandLinkLabel( "\"subreport1\"" );//or set a report name, otherwise no tab is shown by the viewer
            subrep.setOnDemandTabLabel( "\"subreport1\"" );

            // Get subreports engine
            Engine subeng = subrep.getEngine();

            // Get detail area and first detail section of the subreport
            Area subArea = subeng.getArea( "D" );
            Section subSection = subArea.getSection( 0 );

            // Add a text to the subreport
            Text subText = subSection.addText( 2000, 100, 3000, 500 );
            Paragraph subPara = subText.addParagraph();
            subPara.addTextPart( "This picture is in the first subreport." );

            // Add a picture to the subreport
            subSection.addPicture( 1500, 500, 2835, 375, path );

            Subreport subrep2 = section.addSubreport( 1000, 2000, 6000, 700 );
            // subrep2.setOnDemand(true);
            subrep2.setOnDemandLinkLabel( "\"subreport2\"" );//or set a report name, otherwise no tab is shown by the viewer
            subrep2.setOnDemandTabLabel( "\"subreport2\"" );
            // Get subreports engine
            Engine subeng2 = subrep2.getEngine();
            // Get detail area and first detail section of the subreport
            Area subArea2 = subeng2.getArea( "D" );
            Section subSection2 = subArea2.getSection( 0 );
            // Add a text to the subreport
            Text subText2 = subSection2.addText( 2000, 100, 3300, 500 );
            Paragraph subPara2 = subText2.addParagraph();
            subPara2.addTextPart( "This picture is in the second subreport." );
            // Add a picture to the subreport
            subSection2.addPicture( 1500, 500, 2835, 375, path );

            String[] employeeNames =
                { "EmployeeID", "Supervisor", "LastName", "FirstName", "Position", "BirthDate", "HireDate", "HomePhone", "Extension", "ReportsTo",
                    "Salary", "SSN", "EmergencyContactFirstName", "EmergencyContactLastName", "EmergencyContactRelationship", "EmergencyContactPhone" };
            int[] employeeTypes =
                { Field.NUMBER, Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.DATETIME, Field.DATETIME, Field.STRING, Field.STRING,
                    Field.NUMBER, Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.STRING };
            String[] ordersNames =
                { "OrderID", "OrderAmount", "CustomerID", "EmployeeID", "OrderDate", "RequiredDate", "ShipDate", "ShipVia", "Shipped", "PO#",
                    "PaymentReceived" };
            int[] ordersTypes =
                { Field.NUMBER, Field.NUMBER, Field.STRING, Field.NUMBER, Field.DATETIME, Field.DATETIME, Field.DATETIME, Field.STRING,
                    Field.BOOLEAN, Field.STRING, Field.BOOLEAN };
            String[] orderDetailsNames = { "OrderID", "ProductID", "UnitPrice", "Quantity" };
            int[] orderDetailsTypes = { Field.NUMBER, Field.NUMBER, Field.NUMBER, Field.NUMBER };
            String[] productNames = { "ProductID", "ProductName", "Color", "Size", "M/F", "Price", "ProductTypeID", "ProductClass", "SupplierID" };
            int[] productTypes =
                { Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.NUMBER, Field.NUMBER, Field.STRING, Field.NUMBER };

            // Add database connection
            final DatabaseTables dbTables = eng.getDatabaseTables();
            final Fields fields = eng.getFields();

            // Define a data source
            Datasource ds = dbTables.createDatasource( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Define the tables of the data source
            TableSource ts_Employee = ds.createTableSource( "Employees" );
            for( int colIdx = 0; colIdx < employeeNames.length; colIdx++ ) {
                ts_Employee.addColumn( employeeNames[colIdx], employeeTypes[colIdx] );
            }

            TableSource ts_Orders = ds.createTableSource( "Orders" );
            for( int colIdx = 0; colIdx < ordersNames.length; colIdx++ ) {
                ts_Orders.addColumn( ordersNames[colIdx], ordersTypes[colIdx] );
            }

            TableSource ts_OrderDetails = ds.createTableSource( "Order Details" );
            for( int colIdx = 0; colIdx < orderDetailsNames.length; colIdx++ ) {
                ts_OrderDetails.addColumn( orderDetailsNames[colIdx], orderDetailsTypes[colIdx] );
            }

            TableSource ts_Product = ds.createTableSource( "Products" );
            for( int colIdx = 0; colIdx < productNames.length; colIdx++ ) {
                ts_Product.addColumn( productNames[colIdx], productTypes[colIdx] );
            }

            // Add links
            dbTables.addJoin( ts_Employee.getAlias(), "EmployeeID", ts_Orders.getAlias(), "EmployeeID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );
            dbTables.addJoin( ts_Orders.getAlias(), "OrderID", ts_OrderDetails.getAlias(), "OrderID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );
            dbTables.addJoin( ts_OrderDetails.getAlias(), "ProductID", ts_Product.getAlias(), "ProductID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );

            eng.setSF( "{Employees.LastName} = \"Chase\"" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 1 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new SubreportSample().initUI();
    }
}
